// FB Alpha Gaelco World Rally 2 driver module
// Based on MAME driver by Manuel Abadia

#include "tiles_generic2.h"
#include "gaelco.h"
#include "mcs51.h"

static UINT8 *AllMem;
static UINT8 *MemEnd;
static UINT8 *AllRam;
static UINT8 *RamEnd;
static UINT8 *Drv68KROM;
static UINT8 *DrvGfxROM;
static UINT8 *DrvSndROM;
static UINT8 *DrvSprRAM;
static UINT8 *DrvSprBuf;
static UINT8 *DrvPalRAM;
static UINT8 *Drv68KRAM;
static UINT8 *Drv68KRAM2;
static UINT8 *DrvMCUROM;
static UINT8 *DrvMCURAM;
static UINT8 *DrvShareRAM;

static UINT32 *DrvPalette;
static UINT8 DrvRecalc;

static UINT16 *DrvVidRegs;

static UINT8 DrvJoy1[16];
static UINT8 DrvJoy2[16];
static UINT8 DrvJoy3[16];
static UINT8 DrvJoy4[16];
static UINT8 DrvDips[3];
static UINT8 DrvReset;
static UINT16 DrvInputs[4];

static struct BurnInputInfo Wrally2InputList[] = {
	{"P1 Coin",		BIT_DIGITAL,	DrvJoy3 + 8,	"p1 coin"	},
	{"P1 Start",		BIT_DIGITAL,	DrvJoy1 + 7,	"p1 start"	},
	{"P1 Up",		BIT_DIGITAL,	DrvJoy1 + 0,	"p1 up"		},
	{"P1 Down",		BIT_DIGITAL,	DrvJoy1 + 1,	"p1 down"	},
	{"P1 Left",		BIT_DIGITAL,	DrvJoy1 + 3,	"p1 left"	},
	{"P1 Right",		BIT_DIGITAL,	DrvJoy1 + 2,	"p1 right"	},
	{"P1 Button 1",		BIT_DIGITAL,	DrvJoy1 + 4,	"p1 fire 1"	},
	{"P1 Button 2",		BIT_DIGITAL,	DrvJoy1 + 5,	"p1 fire 2"	},
// placeholder for analog inputs
	{"P1 Button 3",		BIT_DIGITAL,	DrvJoy1 + 11,	"p1 fire 3"	},

	{"P2 Coin",		BIT_DIGITAL,	DrvJoy3 + 10,	"p2 coin"	},
	{"P2 Start",		BIT_DIGITAL,	DrvJoy3 + 7,	"p2 start"	},
	{"P2 Up",		BIT_DIGITAL,	DrvJoy3 + 0,	"p2 up"		},
	{"P2 Down",		BIT_DIGITAL,	DrvJoy3 + 1,	"p2 down"	},
	{"P2 Left",		BIT_DIGITAL,	DrvJoy3 + 3,	"p2 left"	},
	{"P2 Right",		BIT_DIGITAL,	DrvJoy3 + 2,	"p2 right"	},
	{"P2 Button 1",		BIT_DIGITAL,	DrvJoy3 + 4,	"p2 fire 1"	},
	{"P2 Button 2",		BIT_DIGITAL,	DrvJoy3 + 5,	"p2 fire 2"	},
// placeholder for analog inputs
	{"P2 Button 3",		BIT_DIGITAL,	DrvJoy2 + 11,	"p2 fire 3"	},

	{"Reset",		BIT_DIGITAL,	&DrvReset,	"reset"		},
	{"Service 1",		BIT_DIGITAL,	DrvJoy4 + 8,	"service"	},
	{"Service 2",		BIT_DIGITAL,	DrvJoy4 + 10,	"service"	},
	{"Service Mode",		BIT_DIGITAL,	DrvJoy4 + 9,	"diag"	},
	{"Dip A",		BIT_DIPSWITCH,	DrvDips + 0,	"dip"		},
	{"Dip B",		BIT_DIPSWITCH,	DrvDips + 1,	"dip"		},
};

STDINPUTINFO(Wrally2)

static struct BurnDIPInfo Wrally2DIPList[]=
{
	{0x16, 0xff, 0xff, 0xdd, NULL				},
	{0x17, 0xff, 0xff, 0xff, NULL				},

	{0   , 0xfe, 0   ,    2, "Service Mode"			},
	{0x16, 0x01, 0x01, 0x01, "Off"				},
	{0x16, 0x01, 0x01, 0x00, "On"				},

	{0   , 0xfe, 0   ,    2, "Coin mechanism"		},
	{0x16, 0x01, 0x02, 0x00, "Common"			},
	{0x16, 0x01, 0x02, 0x02, "Independent"			},
	
	{0   , 0xfe, 0   ,    2, "Demo Sounds"			},
	{0x16, 0x01, 0x04, 0x00, "Off"				},
	{0x16, 0x01, 0x04, 0x04, "On"				},

	{0   , 0xfe, 0   ,    1, "Cabinet 1 Controls"		},
	//{0x16, 0x01, 0x08, 0x00, "Pot Wheel"			},
	{0x16, 0x01, 0x08, 0x08, "Joystick"			},

	{0   , 0xfe, 0   ,    1, "Cabinet 2 Controls"		},
	//{0x16, 0x01, 0x10, 0x00, "Pot Wheel"			},
	{0x16, 0x01, 0x10, 0x10, "Joystick"			},

	{0   , 0xfe, 0   ,    2, "Monitors (change requires Restart!)"			},
	{0x16, 0x01, 0x20, 0x00, "One"				},
	{0x16, 0x01, 0x20, 0x20, "Two"				},
	
	{0   , 0xfe, 0   ,    4, "Difficulty"			},
	{0x16, 0x01, 0xc0, 0x40, "Easy"				},
	{0x16, 0x01, 0xc0, 0xc0, "Normal"			},
	{0x16, 0x01, 0xc0, 0x80, "Hard"				},
	{0x16, 0x01, 0xc0, 0x00, "Hardest"			},

	{0   , 0xfe, 0   ,    2, "Free Play"			},
	{0x17, 0x01, 0x01, 0x01, "Off"				},
	{0x17, 0x01, 0x01, 0x00, "On"				},

	{0   , 0xfe, 0   ,    2, "Credit configuration"		},
	{0x17, 0x01, 0x02, 0x00, "Start 2C/Continue 1C"		},
	{0x17, 0x01, 0x02, 0x02, "Start 1C/Continue 1C"		},

	{0   , 0xfe, 0   ,    8, "Coin B"			},
	{0x17, 0x01, 0x1c, 0x18, "4 Coins 1 Credits"		},
	{0x17, 0x01, 0x1c, 0x10, "3 Coins 1 Credits"		},
	{0x17, 0x01, 0x1c, 0x08, "2 Coins 1 Credits"		},
	{0x17, 0x01, 0x1c, 0x1c, "1 Coin  1 Credits"		},
	{0x17, 0x01, 0x1c, 0x00, "2 Coins 3 Credits"		},
	{0x17, 0x01, 0x1c, 0x0c, "1 Coin  2 Credits"		},
	{0x17, 0x01, 0x1c, 0x14, "1 Coin  3 Credits"		},
	{0x17, 0x01, 0x1c, 0x04, "1 Coin  4 Credits"		},

	{0   , 0xfe, 0   ,    8, "Coin A"			},
	{0x17, 0x01, 0xe0, 0xc0, "4 Coins 1 Credits"		},
	{0x17, 0x01, 0xe0, 0x80, "3 Coins 1 Credits"		},
	{0x17, 0x01, 0xe0, 0x40, "2 Coins 1 Credits"		},
	{0x17, 0x01, 0xe0, 0xe0, "1 Coin  1 Credits"		},
	{0x17, 0x01, 0xe0, 0x00, "2 Coins 3 Credits"		},
	{0x17, 0x01, 0xe0, 0x60, "1 Coin  2 Credits"		},
	{0x17, 0x01, 0xe0, 0xa0, "1 Coin  3 Credits"		},
	{0x17, 0x01, 0xe0, 0x20, "1 Coin  4 Credits"		},
};

STDDIPINFO(Wrally2)

static void  __fastcall wrally2_main_write_word(UINT32 address, UINT16 data)
{
	switch (address)
	{
		case 0x218004:
		case 0x218006:
		case 0x218008:
			DrvVidRegs[(address - 0x218004) / 2] = data;
		return;
	}
}

static UINT16 __fastcall wrally2_main_read_word(UINT32 address)
{
	switch (address)
	{
		case 0x300000:
		case 0x300002:
		case 0x300004:
		case 0x300006:
			return DrvInputs[(address/2) & 3];
	}

	return 0;
}

static UINT8 __fastcall wrally2_main_read_byte(UINT32 address)
{
	switch (address)
	{
		case 0x300000:
		case 0x300001:
		case 0x300002:
		case 0x300003:
		case 0x300004:
		case 0x300005:
		case 0x300006:
			return DrvInputs[(address/2) & 3] >> ((~address & 1) * 8);
	}

	return 0;
}

static void __fastcall gaelco2_sound_write_byte(UINT32 address, UINT8 data)
{
	DrvSprRAM[(address & 0xffff) ^ 1] = data;

	if (address >= 0x202890 && address <= 0x2028ff) {
		// not used.
		return;
	}
}

static void __fastcall gaelco2_sound_write_word(UINT32 address, UINT16 data)
{
	*((UINT16*)(DrvSprRAM + (address & 0xfffe))) = data;

	if (address >= 0x202890 && address <= 0x2028ff) {
		gaelcosnd_w((address - 0x202890) >> 1, data);
		return;
	}
}

static UINT8 __fastcall gaelco2_sound_read_byte(UINT32 address)
{
	if (address >= 0x202890 && address <= 0x2028ff) {
		// not used.
	}
	return DrvSprRAM[(address & 0xffff) ^ 1];
}

static UINT16 __fastcall gaelco2_sound_read_word(UINT32 address)
{
	if (address >= 0x202890 && address <= 0x2028ff) {
		return gaelcosnd_r((address - 0x202890) >> 1);
	}
	return *((UINT16*)(DrvSprRAM + (address & 0xfffe)));
}

static void dallas_sharedram_write(INT32 address, UINT8 data)
{
	if (address >= MCS51_PORT_P0) return;

	if (address >= 0x8000 && address <= 0xffff)
		DrvShareRAM[(address & 0x7fff) ^ 1] = data;

	if (address >= 0x10000 && address <= 0x17fff)
		DrvMCURAM[address & 0x7fff] = data;
}

static UINT8 dallas_sharedram_read(INT32 address)
{
	if (address >= MCS51_PORT_P0) return 0;

	if (address >= 0x8000 && address <= 0xffff)
		return DrvShareRAM[(address & 0x7fff) ^ 1];

	if (address >= 0x10000 && address <= 0x17fff)
		return DrvMCURAM[address & 0x7fff];

	return 0;
}

static void palette_update(INT32 offset)
{
	static const int pen_color_adjust[16] = {
		0, -8, -16, -24, -32, -40, -48, -56, 64, 56, 48, 40, 32, 24, 16, 8
	};

	offset = (offset & 0x1ffe);

	UINT16 color = *((UINT16*)(DrvPalRAM + offset));

	INT32 r = (color >> 10) & 0x1f;
	INT32 g = (color >>  5) & 0x1f;
	INT32 b = (color >>  0) & 0x1f;

	r = (r << 3) | (r >> 2);
	g = (g << 3) | (g >> 2);
	b = (b << 3) | (b >> 2);

	DrvPalette[offset/2] = BurnHighCol(r,g,b,0);

#define ADJUST_COLOR(c) ((c < 0) ? 0 : ((c > 255) ? 255 : c))

	for (INT32 i = 1; i < 16; i++) {
		INT32 auxr = ADJUST_COLOR(r + pen_color_adjust[i]);
		INT32 auxg = ADJUST_COLOR(g + pen_color_adjust[i]);
		INT32 auxb = ADJUST_COLOR(b + pen_color_adjust[i]);

		DrvPalette[0x1000 * i + (offset/2)] = BurnHighCol(auxr,auxg,auxb,0);
	}
}

static void __fastcall gaelco2_palette_write_byte(UINT32 address, UINT8 data)
{
	DrvPalRAM[(address & 0x1fff) ^ 1] = data;
	palette_update(address);
}

static void __fastcall gaelco2_palette_write_word(UINT32 address, UINT16 data)
{
	*((UINT16*)(DrvPalRAM + (address & 0x1ffe))) = data;
	palette_update(address);
}

static INT32 DrvDoReset()
{
	memset (AllRam, 0, RamEnd - AllRam);

	SekOpen(0);
	SekReset();
	SekClose();

	mcs51_reset();

	gaelcosnd_reset();

	return 0;
}

static INT32 MemIndex()
{
	UINT8 *Next; Next = AllMem;

	Drv68KROM	= Next; Next += 0x0100000;

	DrvMCUROM	= Next; Next += 0x0008000;

	DrvPalette	= (UINT32*)Next; Next += 0x10000 * sizeof(UINT32);

	AllRam		= Next;

	DrvSprRAM	= Next; Next += 0x0010000;
	DrvSprBuf	= Next; Next += 0x0010000;
	DrvPalRAM	= Next; Next += 0x0002000;
	Drv68KRAM	= Next; Next += 0x0020000;
	Drv68KRAM2	= Next; Next += 0x0002000;

	DrvShareRAM	= Next; Next += 0x0008000;
	DrvMCURAM	= Next; Next += 0x0008000;

	DrvVidRegs	= (UINT16*)Next; Next += 0x00004 * sizeof(UINT16);

	RamEnd		= Next;

	MemEnd		= Next;

	return 0;
}

static void DrvGfxDecode(UINT8 *tmp, UINT8 *dst)
{
	INT32 size = 0x80000 * 5;

	INT32 Planes[5] = { (size/5)*8*4, (size/5)*8*3, (size/5)*8*2, (size/5)*8*1, (size/5)*8*0 };
	INT32 XOffs[16] = { STEP8(0,1), STEP8(128,1) };
	INT32 YOffs[16] = { STEP16(0,8) };

	GfxDecode2(((size*8)/5)/(16*16), 5, 16, 16, Planes, XOffs, YOffs, 0x100, tmp, dst);
}

static INT32 Wrally2Init()
{
	AllMem = NULL;
	MemIndex();
	INT32 nLen = MemEnd - (UINT8 *)0;
	if ((AllMem = (UINT8 *)BurnMalloc(nLen)) == NULL) return 1;
	memset(AllMem, 0, nLen);
	MemIndex();

	{
		if (BurnLoadRom(Drv68KROM  + 0x000001, 0, 2)) return 1;
		if (BurnLoadRom(Drv68KROM  + 0x000000, 1, 2)) return 1;

		if (BurnLoadRom(DrvMCUROM  + 0x000000, 2, 1)) return 1;

		DrvGfxROM = (UINT8*)BurnMalloc(0x800000);

		UINT8 *tmp = (UINT8*)BurnMalloc(0x80000 * 5);

		BurnLoadRom(tmp + 0x80000 * 0,  3, 1);
		BurnLoadRom(tmp + 0x80000 * 1,  7, 1);
		BurnLoadRom(tmp + 0x80000 * 2, 11, 1);
		BurnLoadRom(tmp + 0x80000 * 3, 13, 1);
		BurnLoadRom(tmp + 0x80000 * 4, 15, 1);

		DrvGfxDecode(tmp, DrvGfxROM + 0x000000);

		BurnLoadRom(tmp + 0x80000 * 0,  4, 1);
		BurnLoadRom(tmp + 0x80000 * 1,  8, 1);
		BurnLoadRom(tmp + 0x80000 * 2, 12, 1);
		BurnLoadRom(tmp + 0x80000 * 3, 14, 1);
		BurnLoadRom(tmp + 0x80000 * 4, 16, 1);

		DrvGfxDecode(tmp, DrvGfxROM + 0x400000);

		BurnFree (tmp);

		DrvSndROM = (UINT8*)BurnMalloc(0x400000);
		memset(DrvSndROM, 0, 0x400000);

		BurnLoadRom(DrvSndROM + 0x100000,  5, 1);
		BurnLoadRom(DrvSndROM + 0x180000,  6, 1);
		BurnLoadRom(DrvSndROM + 0x300000,  9, 1);
		BurnLoadRom(DrvSndROM + 0x380000, 10, 1);
	}

	SekInit(0, 0x68000);
	SekOpen(0);
	SekMapMemory(Drv68KROM,		0x000000, 0x0fffff, SM_ROM);
	SekMapMemory(DrvSprRAM,		0x200000, 0x20ffff, SM_RAM);
	SekMapMemory(DrvPalRAM,		0x210000, 0x211fff, SM_RAM);
	SekMapMemory(Drv68KRAM2, 	0x212000, 0x213fff, SM_RAM);
	SekMapMemory(Drv68KRAM,		0xfe0000, 0xfe7fff, SM_RAM);
	SekMapMemory(DrvShareRAM,	0xfe8000, 0xfeffff, SM_RAM);
	SekSetWriteWordHandler(0,	wrally2_main_write_word);
	SekSetReadWordHandler(0,	wrally2_main_read_word);
	SekSetReadByteHandler(0,	wrally2_main_read_byte);

	SekMapHandler(1,		0x202800, 0x202bff, SM_WRITE | SM_READ);
	SekSetWriteWordHandler(1,	gaelco2_sound_write_word);
	SekSetWriteByteHandler(1,	gaelco2_sound_write_byte);
	SekSetReadWordHandler(1,	gaelco2_sound_read_word);
	SekSetReadByteHandler(1,	gaelco2_sound_read_byte);

	SekMapHandler(2,		0x210000, 0x211fff, SM_WRITE);
	SekSetWriteWordHandler(2,	gaelco2_palette_write_word);
	SekSetWriteByteHandler(2,	gaelco2_palette_write_byte);
	SekClose();

	mcs51_program_data = DrvMCUROM;
	ds5002fp_init(0x69, 0x00, 0x80);
	mcs51_set_write_handler(dallas_sharedram_write);
	mcs51_set_read_handler(dallas_sharedram_read);

	gaelcosnd_start(DrvSndROM, 0 * 0x0200000, 1 * 0x0200000, 0, 0);

	GenericTiles2Init();

	DrvDoReset();

	return 0;
}

static INT32 DrvExit()
{
	GenericTiles2Exit();

	SekExit();

	mcs51_exit();

	BurnFree (AllMem);
	BurnFree (DrvGfxROM);
	BurnFree (DrvSndROM);

	gaelcosnd_exit();

	return 0;
}

static void draw_layer()
{
	INT32 offset = ((DrvVidRegs[0] >> 9) & 0x07) * 0x1000;

	UINT16 *ram = (UINT16*)DrvSprRAM;

	INT32 scrolly = (ram[0x1400] + 0x01 - -16) & 0x1ff;

	if ((DrvVidRegs[0] & 0x8000) == 0)
	{
		INT32 scrollx = (ram[(0x2802)/2] + 0x10 + 4) & 0x3ff;

		for (INT32 offs = 0; offs < 64 * 32; offs++)
		{
			INT32 sx = (offs & 0x3f) * 16;
			INT32 sy = (offs / 0x40) * 16;

			sx -= scrollx;
			if (sx < -15) sx += 1024;
			sy -= scrolly;
			if (sy < -15) sy += 512;

			if (sx >= nScreenWidth2 || sy >= nScreenHeight2) continue;

			INT32 attr0 = ram[offset + (offs * 2) + 0];
			INT32 attr1 = ram[offset + (offs * 2) + 1];

			INT32 code  = (attr1 + ((attr0 & 0x07) << 16)) & 0x7fff;

			INT32 color = (attr0 & 0x7e00) >> 9;
			INT32 flipx = (attr0 & 0x0080) ? 0xf : 0;
			INT32 flipy = (attr0 & 0x0040) ? 0xf : 0;

			if (flipy) {
				if (flipx) {
					Render16x16Tile_Mask_FlipXY_Clip2(pTransDraw2, code, sx, sy, color, 5, 0, 0, DrvGfxROM);
				} else {
					Render16x16Tile_Mask_FlipY_Clip2(pTransDraw2, code, sx, sy, color, 5, 0, 0, DrvGfxROM);
				}
			} else {
				if (flipx) {
					Render16x16Tile_Mask_FlipX_Clip2(pTransDraw2, code, sx, sy, color, 5, 0, 0, DrvGfxROM);
				} else {
					Render16x16Tile_Mask_Clip2(pTransDraw2, code, sx, sy, color, 5, 0, 0, DrvGfxROM);
				}
			}
		}
	}
	else
	{
		for (INT32 sy = 0; sy < nScreenHeight2; sy++)
		{
			INT32 yy = (sy + scrolly) & 0x1ff;
			INT32 yy2 = sy;

			INT32 scrollx = (ram[(0x2802)/2] + 0x10 + 4) & 0x3ff;
			if (DrvVidRegs[0] & 0x8000) {
				scrollx = (ram[(0x2000/2) + yy2] + 0x14) & 0x3ff;
			}

			UINT16 *dst = pTransDraw2 + sy * nScreenWidth2;

			for (INT32 sx = 0; sx < nScreenWidth2 + 16; sx += 16)
			{
				INT32 sxx = (sx + scrollx) & 0x3ff;
				INT32 xx = sx - (scrollx & 0xf);

				INT32 index = ((sxx / 16) + ((yy / 16) * 64)) * 2;

				INT32 attr0 = ram[offset + index + 0];
				INT32 attr1 = ram[offset + index + 1];

				INT32 code  = (attr1 + ((attr0 & 0x07) << 16)) & 0x7fff;

				INT32 color = ((attr0 >> 9) & 0x3f) * 0x20;
				INT32 flipx = (attr0 & 0x80) ? 0xf : 0;
				INT32 flipy = (attr0 & 0x40) ? 0xf : 0;

				UINT8 *gfx = DrvGfxROM + (code * 256) + (((yy & 0x0f) ^ flipy) * 16);

				for (INT32 x = 0; x < 16; x++, xx++)
				{
					if (xx >= 0 && xx < nScreenWidth2) {
						if (gfx[x^flipx]) {
							dst[xx] = gfx[x^flipx] + color;
						}
					}
				}
			}
		}
	}
}

static void draw_sprites()
{
	UINT16 *buffered_spriteram16 = (UINT16*)DrvSprBuf;
	INT32 j, x, y, ex, ey, px, py;

	INT32 start_offset = (DrvVidRegs[1] & 0x10)*0x100;
	INT32 end_offset = start_offset + 0x1000;

	INT32 width = nScreenWidth2;

	INT32 spr_x_adjust = ((width-1) - 320 + 1) - (511 - 320 - 1) - ((DrvVidRegs[0] >> 4) & 0x01);

	for (j = start_offset; j < end_offset; j += 8)
	{
		INT32 data  = buffered_spriteram16[(j/2) + 0];
		INT32 data2 = buffered_spriteram16[(j/2) + 1];
		INT32 data3 = buffered_spriteram16[(j/2) + 2];
		INT32 data4 = buffered_spriteram16[(j/2) + 3];

		INT32 sx = data3 & 0x3ff;
		INT32 sy = data2 & 0x1ff;

		INT32 xflip = data2 & 0x800;
		INT32 yflip = data2 & 0x400;

		INT32 xsize = ((data3 >> 12) & 0x0f) + 1;
		INT32 ysize = ((data2 >> 12) & 0x0f) + 1;

		if (data & 0x8000) continue;

		if (data2 & 0x0200)
		{
			for (y = 0; y < ysize; y++)
			{
				for (x = 0; x < xsize; x++)
				{
					INT32 data5 = buffered_spriteram16[((data4/2) + (y*xsize + x)) & 0x7fff];
					INT32 number = (((data & 0x1ff) << 10) + (data5 & 0x0fff)) & 0x7fff;
					INT32 color = ((data >> 9) & 0x7f) + ((data5 >> 12) & 0x0f);
					INT32 color_effect = ((color & 0x3f) == 0x3f);

					ex = xflip ? (xsize - 1 - x) : x;
					ey = yflip ? (ysize - 1 - y) : y;

					if (color_effect == 0)
					{
						INT32 xx = ((sx + ex*16) & 0x3ff) + spr_x_adjust;

						if (yflip) {
							if (xflip) {
								Render16x16Tile_Mask_FlipXY_Clip2(pTransDraw2, number, xx, ((sy + ey*16) & 0x1ff) + -16, color, 5, 0, 0, DrvGfxROM);
							} else {
								Render16x16Tile_Mask_FlipY_Clip2(pTransDraw2, number, xx, ((sy + ey*16) & 0x1ff) + -16, color, 5, 0, 0, DrvGfxROM);
							}
						} else {
							if (xflip) {
								Render16x16Tile_Mask_FlipX_Clip2(pTransDraw2, number, xx, ((sy + ey*16) & 0x1ff) + -16, color, 5, 0, 0, DrvGfxROM);
							} else {
								Render16x16Tile_Mask_Clip2(pTransDraw2, number, xx, ((sy + ey*16) & 0x1ff) + -16, color, 5, 0, 0, DrvGfxROM);
							}
						}
					} else {
						const UINT8 *gfx_src = DrvGfxROM + number * 0x100;

						for (py = 0; py < 16; py++)
						{
							INT32 ypos = ((sy + ey*16 + py) & 0x1ff) + -16;
							if ((ypos < 0) || (ypos >= nScreenHeight2)) continue;

							UINT16 *srcy = pTransDraw2  + ypos * nScreenWidth2;

							int gfx_py = yflip ? (16 - 1 - py) : py;

							for (px = 0; px < 16; px++)
							{
								INT32 xpos = (((sx + ex*16 + px) & 0x3ff) + spr_x_adjust) & 0x3ff;
								UINT16 *pixel = srcy + xpos;
								INT32 src_color = *pixel;

								INT32 gfx_px = xflip ? (16 - 1 - px) : px;

								INT32 gfx_pen = gfx_src[16*gfx_py + gfx_px];

								if ((gfx_pen == 0) || (gfx_pen >= 16)) continue;

								if ((xpos < 0) || (xpos >= nScreenWidth2)) continue;

								*pixel = src_color + 4096*gfx_pen;
							}
						}
					}
				}
			}
		}
	}
}

static INT32 DualDraw()
{
	if (DrvRecalc) {
		for (INT32 i = 0; i < 0x2000; i+=2) {
			palette_update(i);
		}
		DrvRecalc = 0;
	}

	BurnTransferClear2();

	draw_layer();
	draw_sprites();

	BurnTransferCopy2(DrvPalette);

	return 0;
}

static INT32 DrvFrame()
{
	if (DrvReset) {
		DrvDoReset();
	}

	SekNewFrame();

	{
		memset (DrvInputs, 0xff, 4 * sizeof(UINT16));

		for (INT32 i = 0; i < 16; i++) {
			DrvInputs[0] ^= (DrvJoy1[i] & 1) << i;
			DrvInputs[1] ^= (DrvJoy2[i] & 1) << i;
			DrvInputs[2] ^= (DrvJoy3[i] & 1) << i;
			DrvInputs[3] ^= (DrvJoy4[i] & 1) << i;
		}

		DrvInputs[0] = (DrvInputs[0] & 0xff) | (DrvDips[0] << 8);
		DrvInputs[1] = (DrvInputs[1] & 0xff) | (DrvDips[1] << 8);
	}

	INT32 nInterleave = 256;
	INT32 nCyclesTotal[2] = { (13000000 * 10) / 591, (1000000 * 10) / 591  }; // ?? MHZ @ 59.1 HZ
	INT32 nCyclesDone[2] = { 0, 0 };
	INT32 nSegment = 0;

	SekOpen(0);

	for (INT32 i = 0; i < nInterleave; i++)
	{
		nSegment = (nCyclesTotal[0] - nCyclesDone[0]) / (nInterleave - i);
		nCyclesDone[0] += SekRun(nSegment);

		if (i == 240) SekSetIRQLine(6, SEK_IRQSTATUS_AUTO);

		nCyclesDone[1] += mcs51Run((SekTotalCycles() / 13) - nCyclesDone[1]);
	}

	if (pBurnSoundOut) {
		gaelcosnd_update(pBurnSoundOut, nBurnSoundLen);

		// mono
		for (INT32 i = 0; i < nBurnSoundLen * 2; i+=2)
		{
			pBurnSoundOut[i+0] = pBurnSoundOut[i+1];
		}
	}

	SekClose();

	if (pBurnDraw) {
		BurnDrvRedraw();
	}

	memcpy (DrvSprBuf, DrvSprRAM, 0x10000);

	return 0;
}

static INT32 DrvScan(INT32 nAction, INT32 *pnMin)
{
	struct BurnArea ba;
	
	if (pnMin != NULL) {
		*pnMin = 0x029698;
	}

	if (nAction & ACB_MEMORY_RAM) {
		memset(&ba, 0, sizeof(ba));
		ba.Data	  = AllRam;
		ba.nLen	  = RamEnd-AllRam;
		ba.szName = "All Ram";
		BurnAcb(&ba);
	}

	if (nAction & ACB_DRIVER_DATA) {
		SekScan(nAction);

		mcs51_scan(nAction);

		gaelcosnd_scan(nAction);
	}

	return 0;
}


// World Rally 2: Twin Racing

static struct BurnRomInfo wrally2RomDesc[] = {
	{ "wr2.64",		0x80000, 0x4cdf4e1e, 1 | BRF_PRG | BRF_ESS }, //  0 68k Code
	{ "wr2.63",		0x80000, 0x94887c9f, 1 | BRF_PRG | BRF_ESS }, //  1

	{ "wr2_dallas.bin",	0x08000, 0xe22328c5, 2 | BRF_PRG | BRF_ESS }, //  2 DS5002FP MCU

	{ "wr2.16d",		0x80000, 0xad26086b, 3 | BRF_GRA },           //  3 Graphics & Samples
	{ "wr2.17d",		0x80000, 0xc1ec0745, 3 | BRF_GRA },           //  4
	{ "wr2.18d",		0x80000, 0xe3617814, 3 | BRF_GRA },           //  5
	{ "wr2.19d",		0x80000, 0x2dae988c, 3 | BRF_GRA },           //  6
	{ "wr2.09d",		0x80000, 0x372d70c8, 3 | BRF_GRA },           //  7
	{ "wr2.10d",		0x80000, 0x5db67eb3, 3 | BRF_GRA },           //  8
	{ "wr2.11d",		0x80000, 0xae66b97c, 3 | BRF_GRA },           //  9
	{ "wr2.12d",		0x80000, 0x6dbdaa95, 3 | BRF_GRA },           // 10
	{ "wr2.01d",		0x80000, 0x753a138d, 3 | BRF_GRA },           // 11
	{ "wr2.02d",		0x80000, 0x9c2a723c, 3 | BRF_GRA },           // 12
	{ "wr2.20d",		0x80000, 0x4f7ade84, 3 | BRF_GRA },           // 13
	{ "wr2.13d",		0x80000, 0xa4cd32f8, 3 | BRF_GRA },           // 14
	{ "wr2.21d",		0x80000, 0x899b0583, 3 | BRF_GRA },           // 15
	{ "wr2.14d",		0x80000, 0x6eb781d5, 3 | BRF_GRA },           // 16
};

STD_ROM_PICK(wrally2)
STD_ROM_FN(wrally2)

struct BurnDriver BurnDrvWrally2 = {
	"wrally2", NULL, NULL, NULL, "1995",
	"World Rally 2: Twin Racing\0", NULL, "Gaelco", "Wr2",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_MISC_POST90S, GBF_RACING, 0,
	NULL, wrally2RomInfo, wrally2RomName, NULL, NULL, Wrally2InputInfo, Wrally2DIPInfo,
	Wrally2Init, DrvExit, DrvFrame, DualDraw, DrvScan, &DrvRecalc, 0x10000,
	384, 240, 4, 3
};
